//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.javaClient;

import com.actional.soapapi.Util;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;



/** <!-- ========================================================================================================== -->
* This class is used to prompt the user for input.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class Input
{
	private static final Logger LOGGER = Logger.getLogger(Input.class.getName());

	/** <!-- ================================================================================================== -->
	 * Private default constructor to hide the implicitly public one
	 *
	 * @lastrev fix39953 - Sonar: Add private default constructor to utility classes
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	private Input() {}
	
	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param defaultInput	The default value to return if the client does not enter anything.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static int readInteger(String prompt, Integer defaultInput) throws Exception
	{
		while (true)
		{
			String val = read(prompt, false, defaultInput == null ? null : defaultInput.toString());

			try
			{
				return Integer.parseInt(val);
			}
			catch (Exception e)
			{
				System.out.println("Invalid integer: " + e);
			}
		}
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param defaultInput	The default value to return if the client does not enter anything.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static boolean readBoolean(String prompt, Boolean defaultInput) throws Exception
	{
		while (true)
		{
			String val = read(prompt, false, defaultInput == null ? null : defaultInput.toString());

			try
			{
				return Boolean.valueOf(val).booleanValue();
			}
			catch (Exception e)
			{
				System.out.println("Invalid boolean: " + e);
			}
		}
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for multiple strings. This method keeps prompting until an empty string is entered.
	*
	* @param prompt		The prompt.
	* @param defaultInput	The default value to return if the client does not enter anything.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String[] readStrings(String prompt, String defaultInput) throws Exception
	{
		List values = new ArrayList();
		boolean firstPrompt = true;

		while(true)
		{
			// This is necessary when the agent can not discover all
			// of the endpoints on its own.
			String value = Input.read(prompt, firstPrompt ? defaultInput : "");

			if (Util.isBlank(value))
				break;

			firstPrompt = false;
			values.add(value);
		}

		return (String[])values.toArray(new String[values.size()]);
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String read(String prompt) throws Exception
	{
		return read(prompt, false, null);
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param defaultInput	The default value to return if the client does not enter anything.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String read(String prompt, String defaultInput) throws Exception
	{
		return read(prompt, false, defaultInput);
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param isPassword	True if the entered text should not appear on the console.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String read(String prompt, boolean isPassword) throws Exception
	{
		return read(prompt, isPassword, null);
	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param isPassword	True if the entered text should not appear on the console.
	* @param defaultInput	The default value to return if the client does not enter anything.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String read(String prompt, boolean isPassword, String defaultInput) throws Exception
	{
		// Tweak the prompt if there is a default Input.
		if ((defaultInput != null) && (defaultInput.length() > 0))
		{
			String val = isPassword ? "*****" : defaultInput;
			prompt += " (" + val + "): ";
		}
		else
			prompt +=  ": ";

		String input = doRead(prompt, isPassword);

		if ((input == null) || (input.length() == 0))
		{
			if (defaultInput != null)
				return defaultInput;
		}

		return input;

	}

	/** <!-- ================================================================================================== -->
	* Prompt the user for input.
	*
	* @param prompt		The prompt.
	* @param isPassword	True if the entered text should not appear on the console.
	*
	* @return The input entered by the user.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static String doRead(String prompt, boolean isPassword) throws Exception
	{
		// input holder
		String		input = "";
		MaskingThread	maskingthread = null;

		if (isPassword)
		{
			maskingthread = new MaskingThread(prompt);
			maskingthread.start();
		}

		System.out.print(prompt);

		// block until enter is pressed
		while (true)
		{
			char c = (char)System.in.read();

			if (isPassword)
			{
				// assume enter pressed, stop masking
				maskingthread.stopMasking();
				maskingthread.join();
				// Clear the current line in case the thread
				// wrote a new prompt while we were waiting
				// for it to stop.
				System.out.print("\r                     \r");
			}

			if (c == '\r')
			{
				c = (char)System.in.read();
				if (c == '\n')
					break;
				else
					continue;
			}
			else if (c == '\n')
			{
				break;
			}
			else
			{
				// store the input
				input += c;
			}
		}

		return input;
	}

	/** <!-- ================================================================================================== -->
	* This class attempts to erase characters echoed to the console.
	* It came from: http://java.sun.com/features/2002/09/pword_mask.html
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static class MaskingThread extends Thread
	{
		private boolean	stop = false;
		private int	index;
		private String	prompt;

		/** <!-- ========================================================================================== -->
		* Constructor.
		*
		* @param prompt The prompt displayed to the user.
		* <!-- ----------------------------------------------------------------------------------------- --> */

		public MaskingThread(String prompt)
		{
			this.prompt = prompt;
		}

		/** <!-- ========================================================================================== -->
		* Begin masking until asked to stop.
		* <!-- ----------------------------------------------------------------------------------------- --> */

		public void run()
		{
			while(!stop)
			{
				try
				{
					// attempt masking at this rate
					this.sleep(1);
				}
				catch (InterruptedException iex)
				{
					LOGGER.log(Level.SEVERE, iex.getMessage(), iex);
				}

				if (!stop)
				{
					System.out.print("\r" + prompt + "              \r" + prompt);
				}
				System.out.flush();
			}
		}

		/** <!-- ========================================================================================== -->
		* Instruct the thread to stop masking.
		* <!-- ----------------------------------------------------------------------------------------- --> */

		public void stopMasking()
		{
			this.stop = true;
		}
	}
}
