//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi;

import com.actional.config.BaseData;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.lang.reflect.Array;
import java.net.URL;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Collection;
import java.util.List;
import java.util.HashSet;
import java.util.jar.Manifest;
import java.util.Map;
import java.util.Iterator;
import java.util.Set;
import java.util.jar.Attributes;

/** <!-- ========================================================================================================== -->
* This class provides typical helper methods for manipulating BaseDatas, etc.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class Util
{
	/** <!-- ================================================================================================== -->
	 * The IDObject to use when Building a Policy that matches any object.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	static final com.actional.config.IDObjectHolder IDOBJECT_ANY;

	static
	{
		com.actional.config.IDObjectHolder idObject = new com.actional.config.IDObjectHolder();

		idObject.setPath(new String[] { "*" });

		IDOBJECT_ANY = idObject;
	}

	/** <!-- ================================================================================================== -->
	* BaseData code field flags. The code field in the BaseData class is a bit field.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	/** BaseData's flagged as published are top level and will never go away until,
	* they are explicitly deleted. All BaseData's are unpublished by default. Thus
	* they will be garbage collected as soon as no one is holding a strong
	* reference. */
	public static final int BD_PUBLISHED = 2;

	/** When a BaseData is flagged as a leaf, its KeyID and KeyName fields will be,
	* set, but the other fields will be null. To populate the BaseData (convert to
	* non leaf), the BaseData must be accervated. */
	public static final int BD_LEAF = 4;

	/** If a BaseData referencecs another BaseData that is flagged as null, then the
	* field doing the reference will be set to null. */
	public static final int BD_NULL = 8;

	/** This code value changes the behavior of null values during a set. When this code
	* value is set, null values in the source BaseData overwrite entries in the target
	* instead of treating them as non modified fields (merging). */
	public static final int BD_EXPLICIT_NULL = 16384;
	
	/** <!-- ================================================================================================== -->
	 * Private default constructor to hide the implicitly public one
	 *
	 * @lastrev fix39953 - Sonar: Add private default constructor to utility classes
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	private Util() {}

	/** <!-- ================================================================================================== -->
	* Bit Manipulation for the code field in a BaseData
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static boolean isBitSet(Integer a, int bit)
	{
		if (a == null)
			return false;

		return (a.intValue() & bit) != 0;
	}

	public static boolean isBitSet(BaseData bd, int bit)
	{
		return isBitSet(bd.getCode(), bit);
	}

	public static int bitSet(Integer a, int b)
	{
		if (a == null)
			return b;

		return a.intValue() | b;
	}

	public static void setCode(BaseData bd, int code)
	{
		bd.setCode(new Integer(bitSet(bd.getCode(), code))); //NOSONAR: It is compiled by 1.4
	}

	public static void setAsPublished(BaseData bd)
	{
		setCode(bd, BD_PUBLISHED);
	}

	public static void setAsNull(BaseData bd)
	{
		setCode(bd, BD_NULL);
	}

	public static void setAsExplicitNull(BaseData bd)
	{
		setCode(bd, BD_EXPLICIT_NULL);
	}

	public static void setAsLeaf(BaseData bd)
	{
		setCode(bd, BD_LEAF);
	}

	public static boolean isLeaf(BaseData bd)
	{
		return isBitSet(bd.getCode(), BD_LEAF);
	}

	/** <!-- ================================================================================================== -->
	* Create Leaf BaseDatas
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static BaseData createLeaf(BaseData bd) throws Exception
	{
		BaseData newBD = (BaseData) bd.getClass().newInstance();

		newBD.setKeyID(bd.getKeyID());
		setAsLeaf(newBD);

		return newBD;
	}

	/** <!-- ================================================================================================== -->
	* Retrieve the KeyID for the BaseData. The keyID is serialized for leaf BaseData's so there is
	* no need to accervate the BaseData.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String getKeyID(BaseData bd) throws Exception
	{
		if (bd == null)
			return null;

		return bd.getKeyID();
	}

	/** <!-- ================================================================================================== -->
	* Retrieve the KeyName for the BaseData. The keyName is serialized for leaf BaseData's so there is
	* no need to accervate the BaseData.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String getKeyName(BaseData bd) throws Exception
	{
		if (bd == null)
			return null;

		return bd.getKeyName();
	}

	/** <!-- ================================================================================================== -->
	* Utility functions for checking if an Object is blank.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static boolean isBlank(String s)
	{
		return (s == null || s.length() == 0);
	}

	public static boolean isBlank(BaseData bd)
	{
		return (bd == null || isBitSet(bd, BD_NULL));
	}

	public static boolean isBlank(Object[] array)
	{
		return ((array == null) || (array.length == 0));
	}

	public static boolean isBlank(byte[] array)
	{
		return ((array == null) || (array.length == 0));
	}

	public static boolean isBlank(Collection c)
	{
		return (c == null || c.isEmpty());
	}

	/** <!-- ================================================================================================== -->
	* Utility functions for comparing two strings
	* <!-- ------------------------------------------------------------------------------------------------- --> */
	public static boolean equals(String string1, String string2)
	{
		if ((string1 == null) && (string2 == null))
			return true;
	  	if ((string1 == null) || (string2 == null))
	  		return  false;

	  	return string1.equals(string2);
	}

	/** <!-- ================================================================================================== -->
	* Utility functions for computing the length of items. These functions return 0 if the item is null.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static int length(Object[] array)
	{
		return array == null ? 0 : array.length;
	}

	/** <!-- ================================================================================================== -->
	* Utility functions for computing the size of items. These functions return 0 if the item is null.
	* <!-- ------------------------------------------------------------------------------------------------- --> */
	public static int size(Collection col)
	{
		return col == null ? 0 : col.size();
	}

	/** <!-- ================================================================================================== -->
	* Merges two lists, eliminating any duplicates.
	* ASSUMPTION: both lists are of the same type.
	*
	* @param array1	The first array
	* @param array2	The second array
	*
	* @return The resulting array from the merging of the two given arrays.
	* 
	* @lastrev fix40034 - Sonar: Solving resources improper closing issues
	* <!-- ------------------------------------------------------------------------------------------------- --> */
	public static Object[] merge(final Object[] array1, final Object[] array2)
	{
		if (array1 == null || array1.length == 0)
			return (Object[])array2.clone();

		if (array2 == null || array2.length == 0)
			return (Object[])array1.clone();

		final List a = Arrays.asList(array1);
		final List b = Arrays.asList(array2);

		final Set set = new HashSet();
		set.addAll(a);
		set.addAll(b);

		final Object[] result = (Object[])Array.newInstance(array1[0].getClass(), set.size());
		return set.toArray(result);
	}

	/** <!-- ================================================================================================== -->
	* Extract the key IDs from a list of BaseDatas.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String[] extractKeyIDs(BaseData[] bds)
	{
		if (isBlank(bds))
			return null;

		String[] res = new String[bds.length];

		for (int i = 0; i < bds.length; i++)
			res[i] = bds[i].getKeyID();

		return res;
	}

	/** <!-- ================================================================================================== -->
	* Read a file from Disk into memory.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static byte[] readFileAsBytes(String name) throws IOException
	{
		return readFileAsBytes(new File(name));
	}

	/** <!-- ================================================================================================== -->
	* Read a file into memory as byte[].
	* 
	* @lastrev fix40034 - Sonar: Solving resources improper closing issues
	* <!-- ------------------------------------------------------------------------------------------------- --> */
	public static byte[] readFileAsBytes(final File file) throws IOException
	{
		final FileInputStream is = new FileInputStream(file);
		try
		{
			final int	totalSize = (int) file.length();
			final byte[]	bytes = new byte[totalSize];
			
			int	offset = 0;
			int	left = totalSize;
			
			while (left > 0)
			{
				final int bytesRead = is.read(bytes, offset, left);
	
				if (bytesRead < 0)
				{
					throw new IOException("Prematured end of request. Expected " + totalSize +
								" bytes, received " + offset + " bytes.");
				}
	
				offset += bytesRead;
				left -= bytesRead;
			}
	
			return bytes;
		}
		finally
		{
			is.close();
		}
	}

	/** <!-- ================================================================================================== -->
	* Boolean helper methods
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static boolean booleanValue(Boolean val)
	{
		if (val == null)
			return false;

		return val.booleanValue();
	}

	public static Boolean BooleanValue(boolean val)
	{
		if (val)
			return Boolean.TRUE;
		else
			return Boolean.FALSE;
	}

	/** <!-- ================================================================================================== -->
	* Convert an object to a string. If the object is null, this method returns empty string.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String toString(Object val)
	{
		if (val == null)
			return "";

		return val.toString();
	}

	/** <!-- ================================================================================================== -->
	* This class is used to compare BaseData's based on KeyName.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static final Comparator COMPARATOR_BASEDATA_KEYNAME = new CompareBaseDataKeyName();

	public static class CompareBaseDataKeyName implements Comparator
	{
		public static String getName(Object o)
		{
			if (o == null)
				return "";

			if (((BaseData)o).getKeyName() == null)
				return "";

			return ((BaseData)o).getKeyName();
		}

		public int compare(Object o1, Object o2)
		{
			try
			{
				return getName(o1).compareTo(getName(o2));
			}
			catch(Exception e)
			{
				return 0;
			}
		}
	}

	/** <!-- ================================================================================================== -->
	* This method will load the Manifest that is stored in the jar of the specified class.
	* 
	* @lastrev fix40034 - Sonar: Solving resources improper closing issues
	* <!-- ------------------------------------------------------------------------------------------------- --> */
	public static Manifest loadManifest(Class clazz) throws Exception
	{
		// Convert the class name to a path.
		String className = clazz.getName();

		className = className.replace('.', '/');
		className = "/" + className + ".class";

		// Try and find the full URL for the class.
		URL url = clazz.getResource(className);
		if (url == null)
			throw new Exception("Failed to find resource: " + className);

		String	strUrl = url.toString();

		if (!strUrl.endsWith(className))
			throw new Exception("Resource URL for '" + strUrl + "' does not end with: " + className);

		strUrl = strUrl.substring(0, strUrl.length() - className.length());

		// Build the URL to the manifest.
		strUrl += "/META-INF/MANIFEST.MF";

		URL manifestUrl = new URL(strUrl);

		InputStream inStream = manifestUrl.openStream();

		if (inStream == null)
			throw new Exception("Failed to load resource: " + strUrl);

		return new Manifest(inStream);
	}

	/** <!-- ================================================================================================== -->
	* This method with get the specification version of the actional-taskapi.jar.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String getTaskAPISpecificationVersion() throws Exception
	{
		// Pass a class that we know is in that JAR.
		return getSpecificationVersion(BaseData.class);
	}

	/** <!-- ================================================================================================== -->
	* This method will get the specification version from the manifest where the specified class is
	* stored. <br>
	* <br>
	* <b>Note:<b> If a jar is rebuilt it may not contain a manifest with the correct information.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static String getSpecificationVersion(Class clazz) throws Exception
	{
		Manifest	manifest = loadManifest(clazz);
		Map		entries = manifest.getEntries();
		Iterator	entryKeyIterator = null;

		if (entries != null)
		{
			Set entryKeys = entries.keySet();

			if (entryKeys != null)
				entryKeyIterator = entryKeys.iterator();
		}

		if (entryKeyIterator != null)
		{
			while (entryKeyIterator.hasNext())
			{
				String name = (String)entryKeyIterator.next();

				if ((name != null) && (name.startsWith("com/actional/")))
				{
					Attributes attribs = manifest.getAttributes(name);
					return attribs.getValue("Specification-Version");
				}
			}
		}

		return null;
	}

}
