//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.server;

// From the Task API
import com.actional.config.BaseData;
import com.actional.lg.db.AgentConfig;
import com.actional.lg.db.LGNode;
// From the SDK
import com.actional.soapapi.Util;
import com.actional.task.to.ProcessingMsgTO;
import com.actional.task.type.ProcessingMsgType;
import com.actional.lg.server.broker.ProvisionInput;

/** <!-- ========================================================================================================== -->
* This class provides log functionality for the SDK.
*
* @lastrev fix40865 - remove AgentConfig / AgentConfigReference / AgentInfo from AMS config mgr
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class AgentConfigTask extends ServerTask
{
	/** This tunning mask for agents processes all events so that none are lost
	* when the agent can not keep up. */
	public static final Integer	AGENT_TUNNING_PROCESS_ALL = new Integer(0x00000000); //NOSONAR: It is compiled by 1.4

	/** This tunning mask for agents discards all events (Server, Client, Low priority)
	* when the FDR is overloaded. This setting is good when it is more important
	* to keep the instrumented application running and monitoring statistics / policies
	* can be lost. */
	public static final Integer	AGENT_TUNNING_OVERFLOW_DISCARD = new Integer(0x00001C00); //NOSONAR: It is compiled by 1.4

	/** <!-- ================================================================================================== -->
	 * Locate the ID of the specified agent given a node name.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	public String locateAgentID(String nodeName) throws Exception
	{
		AgentConfig	agent = new AgentConfig();
		LGNode[]	nodes = new LGNode[1];
		LGNode		node = new LGNode();

		node.setKeyName(nodeName);
		node.setIsFlowMap(Boolean.FALSE);
		nodes[0] = node;
		agent.setLGNodes(nodes);

		String[] matches = config().configTask.findAllKeys(agent);
		int	numMatches = Util.length(matches);

		if (numMatches == 0)
			throw new Exception("Failed to locate node with name: " + nodeName);
		else if (numMatches > 1)
			throw new Exception("Multiple nodes with the same name.");
		else
		{
			String match = matches[0];

			log().debug(1, "Found agent ID: " + match);

			return match;
		}
	}

	/** <!-- ================================================================================================== -->
	 * Locate the specified node given the node name.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	public LGNode locateNode(String nodeName) throws Exception
	{
		// The following code is more efficient than calling ILGCallSiteTask.getAllLGNodes();

		LGNode node = new LGNode();

		node.setKeyName(nodeName);
		node.setIsFlowMap(Boolean.FALSE);

		BaseData[] matches = config().configTask.find(node);
		int	numMatches = Util.length(matches);

		if (numMatches > 1)
			throw new Exception("There are multiple matching nodes.");
		else if (numMatches == 1)
		{
			node = (LGNode)matches[0];
			log().debug(1, "Found node ID: " + node.getKeyID());
			return node;
		}
		else
			return null;
	}

	/** <!-- ================================================================================================== -->
	 * TRUE if the specified node is managed.
	 *
	 * @lastrev fix40865 - remove AgentConfig / AgentConfigReference / AgentInfo from AMS config mgr
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	public boolean isManaged(LGNode node) throws Exception
	{
		return node.getIsManaged();
	}

	/** <!-- ================================================================================================== -->
	 * Provision the specified agent.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public boolean provisionAgent(String agentID) throws Exception
	{
		if (agentID == null)
		{
			log().debug(1, "Agent to provision could not be found.");
			return false;
		}

		// This will block until the specified agent is provisioned.
		final ProvisionInput input = new ProvisionInput();

		input.setMemberIDs(new String[] { agentID });
		input.setCompletionTimeout(20L * 1000L);

		config().activeAgentTask.provision(input);

		return true;
	}

	/** <!-- ================================================================================================== -->
	 * Check all of the nodes in the system for duplicate endpoints, etc.
	 *
	 * @return TRUE if configuration is valid
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public boolean checkNodes() throws Exception
	{
		log().info("Checking all nodes for configuration problems...");

		// check all nodes
		ProcessingMsgTO  res = config().lgCallSiteTask.checkNodes(null);

		if (res.getSeverity() == ProcessingMsgType.OK)
		{
			log().info("PASSED");
			return true;
		}

		log().info("FAILED: " + res.getMessage());

		for (int i = 0, s = Util.length(res.getDetails()); i < s; i++)
		{
			ProcessingMsgTO detail = res.getDetails()[i];

			log().info(1, detail.getSeverity() + " " + detail.getMessage());
		}

		return false;
	}
}
