//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.soapstation;

import com.actional.soapstation.task.*;
import com.actional.ws.db.*;
import com.actional.soapstation.db.*;
import com.actional.config.Blob;
import com.actional.soapapi.Util;

/** <!-- ========================================================================================================== -->
* Helper methods for creating, groups.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class WsdlTask extends SSTTask
{
	/** <!-- ================================================================================================== -->
	* Parse the contents of the WSDL at the specified URL.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public WsdlContentDef parseWsdl(String urlToWsdl) throws Exception
	{
		// Parse and extract all the ports.
		ParseOptions options = new ParseOptions();
		options.setMetadata(new WsMetadata());
		options.getMetadata().setLocation(urlToWsdl);
		options.setReturnMetadata(true);

		// No need to parse the schemas in the WSDL
		options.setReturnXmlTypes(false);

		return config().wsdlTask.parse(options);
	}

	/** <!-- ================================================================================================== -->
	* Load the specified File and ask SST to parse it.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public WsdlContentDef parseWsdl(byte[] content) throws Exception
	{
		// Parse and extract all the ports.

		Blob wsdlContent = new Blob();
		wsdlContent.setBytes(content);

		WsResource wsresource = new WsResource();
		wsresource.setContent(wsdlContent);

		WsMetadata wsmetadata = new WsMetadata();
		wsmetadata.setResources(new WsResource[1]);
		wsmetadata.getResources()[0] = wsresource;

		ParseOptions options = new ParseOptions();
		options.setMetadata(wsmetadata);

		// No need to have the server return the content since we
		// already have it. This saves on network bandwith
		options.setReturnMetadata(true);

		// No need to parse the schemas in the WSDL
		options.setReturnXmlTypes(false);

		WsdlContentDef contentDef = config().wsdlTask.parse(options);

		return contentDef;
	}

	/** <!-- ================================================================================================== -->
	* This structure of Binding / Port pairs.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static class WsdlBindingPortDef
	{
		public WsdlBindingDef	binding;
		public WsdlPortDef	port;

		public WsdlBindingPortDef(WsdlBindingDef binding, WsdlPortDef port)
		{
			this.binding = binding;
			this.port = port;
		}
	}

	/** <!-- ================================================================================================== -->
	* Expand all the bindings so that it is easier to choose one.
	*
	* @return {@link java.util.List} of {@link WsdlBindingPortDef}
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public java.util.List expandPortsFromBindings(WsdlBindingDef[] bindings) throws Exception
	{
		int len = Util.length(bindings);

		if (len == 0)
			return null;

		java.util.List res = new java.util.ArrayList();

		for (int i = 0; i < len; i++)
		{
			WsdlBindingDef	binding = bindings[i];
			WsdlPortDef[]	ports = binding.getPorts();

			for (int j = 0, js = Util.length(ports); j < js; j++)
			{
				WsdlPortDef port = ports[j];
				res.add(new WsdlBindingPortDef(binding, port));
			}
		}

		return res;
	}

	/** <!-- ================================================================================================== -->
	* Build the endpoint from the WSDL port.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public Endpoint buildEndpoint(WsdlPortDef portDef) throws Exception
	{
		if (portDef instanceof WsdlHttpPortDef)
		{
			Endpoint endpoint = new Endpoint();

			endpoint.setAddress(((WsdlHttpPortDef)portDef).getLocation());

			return endpoint;
		}
		else if (portDef instanceof WsdlJmsPortDef)
		{
			WsdlJmsPortDef	jmsPortDef = (WsdlJmsPortDef)portDef;

			if (jmsPortDef.getWsifAddress() != null)
			{
				WsifJmsAddress	address = jmsPortDef.getWsifAddress();
				JmsEndpoint	endpoint = new JmsEndpoint();

				endpoint.setInitCxtFact(address.getInitCxtFact());
				endpoint.setJndiProvURL(address.getJndiProvURL());
				endpoint.setDestStyle(address.getDestStyle());
				endpoint.setJndiConnFactName(address.getJndiConnFactName());
				endpoint.setJndiDestName(address.getJndiDestName());
				endpoint.setProperties(address.getProperties());

				return endpoint;
			}
		}

		throw new Exception("Failed to build endpoint: " + portDef);
	}
}

