//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.soapstation;

import java.util.List;

import com.actional.soapapi.Util;
import com.actional.soapstation.db.AccessPointNode;
import com.actional.soapstation.db.Application;
import com.actional.soapstation.db.Endpoint;
import com.actional.soapstation.db.JmsTransport;
import com.actional.soapstation.db.ManagedServiceNode;
import com.actional.soapstation.db.SvcGroup;
import com.actional.soapstation.db.Transport;
import com.actional.soapstation.db.WsdlBinding;
import com.actional.soapstation.db.XmlProcessorLC;
import com.actional.soapstation.task.WsdlBindingDef;
import com.actional.soapstation.task.WsdlContentDef;
import com.actional.soapstation.task.WsdlPortDef;

/** <!-- ========================================================================================================== -->
* Helper methods for creating, groups.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class ServiceTask extends SSTTask
{
	/** <!-- ================================================================================================== -->
	* Create a pass through service group.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public static class GroupCreationInfo
	{
		public String groupName;
		public String category;
		public String uniqueKey;
		public String wsdlUrl;
		public String applicationName;
		public String listenerName;
		public String senderName;

		/** If non-null, the Access Point will only expose the specified operations. */
		public List apOperations;

		private boolean ignoreProxy = true;
		public void setIgnoreProxy(boolean val)	{ignoreProxy = val;}
		public boolean getIgnoreProxy()			{return ignoreProxy;}
	}

	public String createPassThroughGroup(GroupCreationInfo info) throws Exception
	{
		if (info == null)
			return null;

		Application	app = (Application)findFirst(Application.class, info.applicationName);
		Transport	listener = (Transport)findFirst(Transport.class, info.listenerName);
		Transport	sender = (Transport)findFirst(Transport.class, info.senderName);

		return createPassThroughGroup(	info,
						app,
						listener,
						sender,
						null);
	}

	/** <!-- ================================================================================================== -->
	* Parse and extract first port from WSDL
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected static class WsdlBindingPortDef
	{
		WsdlBinding	binding;
		WsdlPortDef	portDef;
	}

	protected WsdlBindingPortDef firstWsdlBinding(String urlToWsdl, GroupCreationInfo info) throws Exception
	{
		WsdlContentDef		wsdlContentDef = ((SdlTask)createTask(SdlTask.class)).parseSDL(urlToWsdl, info.getIgnoreProxy());
		WsdlBindingDef[]	bindingDefs = wsdlContentDef.getBindings();
		WsdlBindingDef		bindingDef = bindingDefs[0];
		WsdlPortDef[]		portDefs = bindingDef.getPorts();

		// Choose the first Port
		WsdlPortDef		portDef = portDefs[0];

		System.out.println("Selecting port: " + portDef.getName());

		// Build a WSDLBinding

		WsdlBinding	wsdlBinding = new WsdlBinding();

		wsdlBinding.setQName(bindingDef.getQName());
		wsdlBinding.setMetadata(wsdlContentDef.getMetadata());

		WsdlBindingPortDef res = new WsdlBindingPortDef();
		res.binding = wsdlBinding;
		res.portDef = portDef;
		return res;
	}

	/** <!-- ================================================================================================== -->
	* Create a pass through service group with the specified settings.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public String createPassThroughGroup(	GroupCreationInfo	info,
						Application		msApp,
						Transport		apListener,
						Transport		msSender,
						XmlProcessorLC		xmlProcessor)
		throws Exception
	{
		if (info == null)
			return null;

		String accessPointUniqueKey = info.uniqueKey;
		String urlToWsdl = info.wsdlUrl;

		//------------------------------------------------------
		// Parse the WSDL and select the first Binding
		//------------------------------------------------------

		WsdlBindingPortDef 	wsdlInfo = firstWsdlBinding(urlToWsdl, info);

		//------------------------------------------------------
		// Check if we are missing any information and fill
		// in defaults.
		//------------------------------------------------------

		if (Util.isBlank(accessPointUniqueKey))
			accessPointUniqueKey = info.groupName;

		if (Util.isBlank(msApp))
			msApp = ((ApplicationTask)createTask(ApplicationTask.class)).getDefaultApplication();

		if (Util.isBlank(msApp))
		{
			// We still don't know which application. Create a new one.
			msApp = new Application();

			String appName = info.groupName;

			if (Util.isBlank(appName))
			{
				// Default to the name of the Port from the WSDL.
				appName = wsdlInfo.binding.getQName().getLocalName();
			}

			msApp.setKeyName(appName);
			msApp = (Application)config()._getLeaf(config().appTask.createApplication(msApp));
		}

		if (Util.isBlank(apListener))
			apListener = ((TransportsTask)createTask(TransportsTask.class)).defaultListener();

		if (Util.isBlank(msSender))
			msSender = ((TransportsTask)createTask(TransportsTask.class)).defaultSender();


		//------------------------------------------------------
		// Prepare the data structures for the Service
		// group creation.
		//------------------------------------------------------

		// Group
		SvcGroup svcGroup = new SvcGroup();

		svcGroup.setKeyName(info.groupName);
		svcGroup.setCategory(info.category);

		// Access Point
		AccessPointNode apNode = new AccessPointNode();
		apNode.setTransport(apListener);
		apNode.setUniqueKey(accessPointUniqueKey);
		apNode.setDescription("Initial description");

		// Endpoint
		Endpoint endpoint = new Endpoint();
		endpoint.setTransport(msSender);
		if (!(msSender instanceof JmsTransport))
		{
			int i = urlToWsdl.lastIndexOf('?');
			if (i == -1)
				endpoint.setAddress(urlToWsdl);
			else
				endpoint.setAddress(urlToWsdl.substring(0, i));
		}

		// Managed Service
		ManagedServiceNode msNode = new ManagedServiceNode();
		msNode.setEndpoint(endpoint);

		//------------------------------------------------------
		// Create the Group.
		//------------------------------------------------------

		String svcGroupID = config().serviceTask.addSimpleServiceGroup(wsdlInfo.binding, wsdlInfo.portDef,
					new Object[] {		svcGroup,
								apNode,
								msNode,
								msApp,
								xmlProcessor,
								info.apOperations} );

		// Activate the SvcGroup right away.
		config().serviceTask.activateServiceGroupRevision(svcGroupID);
		return svcGroupID;
	}
}
