//=====================================================================================================================
// Copyright (c) 2016-2019. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.javaClient;

// Java SDK
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.actional.config.BaseData;
import com.actional.config.Blob;
import com.actional.db.QName;
import com.actional.soapapi.Util;
import com.actional.soapapi.soapstation.SSTConfig;
import com.actional.soapapi.soapstation.SdlTask;
import com.actional.soapapi.soapstation.WsdlTask;
import com.actional.soapstation.db.AccessPointNode;
import com.actional.soapstation.db.Application;
import com.actional.soapstation.db.AuthenticatorLC;
import com.actional.soapstation.db.CertificateEncodingType;
import com.actional.soapstation.db.CertificateEntry;
import com.actional.soapstation.db.Credentials;
import com.actional.soapstation.db.Endpoint;
import com.actional.soapstation.db.HTTPListener;
import com.actional.soapstation.db.HttpSender;
import com.actional.soapstation.db.HttpTransport;
import com.actional.soapstation.db.HttpsTransport;
import com.actional.soapstation.db.JavaPluginLC;
import com.actional.soapstation.db.JavaPluginType;
import com.actional.soapstation.db.ManagedServiceNode;
import com.actional.soapstation.db.SSTRole;
import com.actional.soapstation.db.SvcGroup;
import com.actional.soapstation.db.SvcGroupRevisions;
import com.actional.soapstation.db.Transport;
import com.actional.soapstation.db.UserEntity;
import com.actional.soapstation.db.WsdlBinding;
import com.actional.soapstation.db.XmlProcessorLC;
import com.actional.soapstation.task.AccessPointDef;
import com.actional.soapstation.task.CertificateDef;
import com.actional.soapstation.task.DirectoryFilterType;
import com.actional.soapstation.task.GeneratedWsdlContentDef;
import com.actional.soapstation.task.OperationSelection;
import com.actional.soapstation.task.ServiceDescriptionLanguageParseInput;
import com.actional.soapstation.task.WsdlBindingDef;
import com.actional.soapstation.task.WsdlContentDef;
import com.actional.soapstation.task.WsdlPortDef;
import com.actional.ws.db.WsMetadata;
import com.actional.ws.db.WsResource;
import org.w3c.dom.Document;

/** <!-- ========================================================================================================== -->
* This is an example of using the SDK to perform different operations.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class SSTClient extends ClientBase
{
	private static final Logger LOGGER = Logger.getLogger(SSTClient.class.getName());

	/** <!-- ================================================================================================== -->
	 * Constructor
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public SSTClient(SSTConfig sstConfig)
	{
		super(sstConfig);
	}

	/** <!-- ================================================================================================== -->
	 * Retrieve the config object.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected SSTConfig getConfig()
	{
		return (SSTConfig) itsConfig;
	}

	/** <!-- ================================================================================================== -->
	* Helper function to dump an error message to the console.
	*
	* @param msg	The message to display.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void failed(String msg)
	{
		info("Failed: " + msg);
	}

	/** <!-- ================================================================================================== -->
	 * Retrieve SdlTask which is a client side class for simplifying certain WSDL/WADL operations.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	protected SdlTask getSdlAPITask() throws Exception
	{
		return (SdlTask) getConfig().createTask(SdlTask.class);
	}

	/** <!-- ================================================================================================== -->
	 * This is the method that performs the different actions.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public void test() throws Exception
	{
		info("Performing tests...");
		info("");

		displayVersion();

		displayKeyCertificates();

		String appID = createApplication();
		listApplications();

		String webserviceID = createWebService();
		updateWebservice(webserviceID);

		listSvcGroups();
		findServiceGroup();

		deleteExamples(webserviceID);

		String restServiceID = createRESTService();

		listSvcGroups();
		findServiceGroup();

		deleteExamples(restServiceID);

		info("DONE");
	}

	/** <!-- ================================================================================================== -->
	 * Delete a few items that have been created.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	protected void deleteExamples(String serviceGroupID) throws Exception
	{
		deleteExamples(serviceGroupID, false);
	}

	protected void deleteExamples(String serviceGroupID, boolean deleteApplication) throws Exception
	{
		// service group
		info("---------");
		info("Deleting the Service Group we created:" + serviceGroupID);

		// This will delete the specified SvcGroup and all its revisions.
		getConfig().serviceTask.deleteServiceGroupRevisions(serviceGroupID);

		// This will delete the specified SvcGroup revision. If there are no
		// other revisions then the SvcGroup will be gone.
		// getConfig().serviceTask.deleteServiceGroup(serviceGroupID);

		if (deleteApplication)
		{
			info("---------");
			info("Delete application:");
			String apID = selectApplication("Select application to delete").getKeyID();
			getConfig().appTask.deleteApplication(apID);
		}
	}

	/** <!-- ================================================================================================== -->
	 * Create a new application
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected String createApplication() throws Exception
	{
		info("---------");
		info("Creating application:");

		Application newApp = new Application();

		newApp.setKeyName(Input.read("Application name", "Default Application"));
		newApp.setDescription(Input.read("Application description", "Default Application"));

		return getConfig().appTask.createApplication(newApp);
	}

	/** <!-- ================================================================================================== -->
	 * Select one application from the product.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected Application selectApplication(String prompt) throws Exception
	{
		Application[] applications = getConfig().appTask.getAllApplications();
		getConfig().acervate(applications);
		Arrays.sort(applications, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(applications); i < size; i++)
		{
			Application app = applications[i];
			String message = i + ") " + app.getKeyName();

			// This is how we check if a particular BD can be written.
			if (getConfig().canWrite(app))
				message += " - writeable";

			info(message);
		}

		int choice = Input.readInteger(prompt, new Integer(0)); //NOSONAR: It is compiled by 1.4
		return applications[choice];
	}

	/** <!-- ================================================================================================== -->
	 * This will list all the applications that the user can read.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listApplications() throws Exception
	{
		info("---------");
		info("Select Application to display its details:");

		displayApplication(selectApplication("Details for application"));
	}

	/** <!-- ================================================================================================== -->
	 * Create a new application
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void displayApplication(String appID) throws Exception
	{
		displayApplication(getConfig().appTask.retrieveApplication(appID));
	}

	/** <!-- ================================================================================================== -->
	 * Create a new application
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void displayApplication(Application app) throws Exception
	{
		app = (Application) getConfig().acervate(app);
		infoField("Name", app.getKeyName());
		infoField("Description", app.getDescription());
	}

	/** <!-- ================================================================================================== -->
	 * Create WebService
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listExistingTransport() throws Exception
	{
		info("---------");
		info("Existing transports: ");

		Transport transport = selectExistingTransport("Select transport");
		// Display details for the selected transport.
	}

	/** <!-- ================================================================================================== -->
	 * Create WebService
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected Transport selectExistingTransport(String prompt) throws Exception
	{
		Transport[] transports = getConfig().transportsTask.getAllTransports();
		getConfig().acervate(transports);
		Arrays.sort(transports, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(transports); i < size; i++)
		{
			Transport t = transports[i];
			String message = i + ") " + t.getKeyName();

			if (t instanceof HttpTransport)
			{
				HttpTransport ht = (HttpTransport) t;
				HTTPListener l = (HTTPListener) getConfig().acervate(ht.getListener());

				if (l != null && !Util.booleanValue(l.getIsDisabled()))
					message += " - Listener";

				HttpSender s = (HttpSender) getConfig().acervate(ht.getSender());

				if (s != null && !Util.booleanValue(s.getIsDisabled()))
					message += " - Sender";
			}
			else if (t instanceof HttpsTransport)
			{
				HttpsTransport ht = (HttpsTransport) t;
				HTTPListener l = (HTTPListener) getConfig().acervate(ht.getListener());

				if (l != null && !Util.booleanValue(l.getIsDisabled()))
					message += " - Listener";

				HttpSender s = (HttpSender) getConfig().acervate(ht.getSender());

				if (s != null && !Util.booleanValue(s.getIsDisabled()))
					message += " - Sender";
			}
			else
			{
				failed("Unexpected transport type: " + t.getClass().getName());
			}

			info(message);
		}

		int choice = Input.readInteger(prompt, new Integer(0)); //NOSONAR: It is compiled by 1.4
		return transports[choice];
	}

	/** <!-- ================================================================================================== -->
	 * This will list all the Service Groups that the user can read.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listSvcGroups() throws Exception
	{
		info("---------");
		info("All Service Groups:");

		SvcGroup[] svcGroups = getConfig().serviceTask.getActiveOrMostRecentRevisions();
		getConfig().acervate(svcGroups);
		Arrays.sort(svcGroups, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(svcGroups); i < size; i++)
		{
			SvcGroup group = svcGroups[i];
			String message = i + ") " + group.getKeyName();

			if (getConfig().serviceTask.isServiceGroupActive(group.getKeyID()))
				message += " - active";

			if (getConfig().serviceTask.isServiceGroupLocked(group.getKeyID()))
				message += " - locked";

			info(message);
		}

		int choice = Input.readInteger("Details for SvcGroup", new Integer(0)); //NOSONAR: It is compiled by 1.4
		displaySvcGroup(svcGroups[choice]);
	}

	/** <!-- ================================================================================================== -->
	 * This will list all the Service Groups that the user can read.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void displaySvcGroup(SvcGroup group) throws Exception
	{
		group = (SvcGroup) getConfig().acervate(group);

		// Display information about the group as a whole.

		infoField("Name", group.getKeyName());
		infoField("Description", group.getDescription());
		infoField("Revision Number", group.getRevisionNumber());
		infoField("Revision Comment", group.getRevisionComment());
		infoField("Version Number", group.getVersionNumber());
		infoField("Version Comment", group.getVersionComment());

		// Display information about each access point in the group.
		AccessPointNode[] aps = getConfig().serviceTask.getAccessPoints(group.getKeyID());
		getConfig().acervate(aps);
		Arrays.sort(aps, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, s = Util.length(aps); i < s; i++)
		{
			AccessPointNode ap = aps[i];
			infoField("AccessPoint", ap.getKeyName());
			infoField(2, "UniqueKey", ap.getUniqueKey());
			infoField(2, "Documentation", ap.getDocumentation());
			infoField(2, "Listener", Util.getKeyName(ap.getTransport()));
			infoField(2, "Client WSDL/WADL", getConfig().apTask.getWsdlFileUrl(ap.getKeyID()));
			infoField(2, "Endpoint", getConfig().apTask.getSoapEndpointUrl(ap.getKeyID()));

			// list information about the security policy.
			displaySdl(2, ap.getWsdlBinding());
			displayAPSvcOperations(ap);

			if (getConfig().sdlTask.isRESTDescriptionLanguage(ap.getWsdlBinding()))
				return;

			// display the WSDL content
			GeneratedWsdlContentDef wsdlContent = getConfig().apTask.generateWSDL(ap.getKeyID());

			if (wsdlContent != null)
			{
				Blob content = wsdlContent.getContent();

				if (content != null)
				{
					byte[] bytes = content.getBytes();
					if (bytes != null)
					{
						infoField(2, "WSDLContent", new String(bytes));
					}
				}
			}
		}

		// Display information about each managed service in the group.
		ManagedServiceNode[] mss = getConfig().serviceTask.getManagedServices(group.getKeyID());
		getConfig().acervate(mss);
		Arrays.sort(mss, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, s = Util.length(mss); i < s; i++)
		{
			ManagedServiceNode ms = mss[i];
			infoField("ManagedService", ms.getKeyName());
			infoField(2, "Application", Util.getKeyName(ms.getApplication()));

			Endpoint endpoint = (Endpoint) getConfig().acervate(ms.getEndpoint());

			infoField(2, "Sender", Util.getKeyName(endpoint.getTransport()));
			infoField(2, "Endpoint", endpoint.getAddress());
			displaySdl(2, ms.getWsdlBinding());
		}
	}

	/** <!-- ================================================================================================== -->
	 * This will dump information about the WSDL/WADL.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void displaySdl(int level, WsdlBinding wsdlBinding) throws Exception
	{
		wsdlBinding = (WsdlBinding) getConfig().acervate(wsdlBinding);
		infoField(2, "WSDL/WADL Binding Name", toString(wsdlBinding.getQName()));

		WsMetadata metadata = (WsMetadata) getConfig().acervate(wsdlBinding.getMetadata());
		infoField(2, "WSDL/WADL File From", metadata.getLocation());
		// WSDescription.getMetadata() contains the Web Service metadata (schemas, WSDLs, policies)
	}

	/** <!-- ================================================================================================== -->
	 * This will dump operations on the AP
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	protected void displayAPSvcOperations(AccessPointNode apnode) throws Exception
	{
		// use the wsdlbinding, and allow the task to parse the wsdl
		apnode = (AccessPointNode) getConfig().acervate(apnode);
		final OperationSelection[] ops = getConfig().serviceTask.getAccessPointOperationSelection(apnode);

		for (int i = 0, maxi = ops.length; i < maxi; i++)
		{
			if (ops[i].isSelected())
				infoField(3, "Operation: ", ops[i].getOperation().getKeyName());
		}
	}

	/** <!-- ================================================================================================== -->
	 * Select a list of operation from the given WSDL/WADL binding
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	protected String[] selectSvcOperations(WsdlBinding wsdlBinding) throws Exception
	{
		if (wsdlBinding != null)
		{
			ArrayList selectedOps = new ArrayList();

			final AccessPointNode apnode = new AccessPointNode();
			apnode.setWsdlBinding(wsdlBinding);

			final OperationSelection[] ops = getConfig().serviceTask
					.getAccessPointOperationSelection(apnode);

			for (int i = 0, maxi = ops.length; i < maxi; i++)
			{
				infoField(2, "Operation: ", ops[i].getOperation().getKeyName());

				if ("y".equals(Input.read("Select this operation?", "y")))
					selectedOps.add(ops[i].getOperation().getKeyName());
			}

			return (String[]) selectedOps.toArray(new String[selectedOps.size()]);
		}

		return null;
	}

	protected WsdlBinding constructSdlBinding(String wsdlLocator, String wsdlLocatorType,
			String wsdlBindingQNameLocalName, String wsdlBindingQNameXmlNs, boolean fullLogging,
			boolean isWadl) throws Exception
	{

		info("");
		info("--------------");
		info("Constructing WsdlBinding");
		info("--------------");
		info("");

		Blob wsdlBindingWsMetadataResourceBlob = new Blob();

		String wsdlBindingWsMetadataResourceBlobString = null;

		if (wsdlLocator != null)
		{
			if (wsdlLocator.length() > 0)
			{
				if ("URL".equalsIgnoreCase(wsdlLocatorType))
				{
					info("WSDL defined at the following URL : " + wsdlLocator);

					try
					{
						URL url = new URL(wsdlLocator);

						InputStream in = new BufferedInputStream(url.openStream());
						ByteArrayOutputStream out = new ByteArrayOutputStream();

						byte[] buf = new byte[1024];
						int n = 0;

						while (-1 != (n = in.read(buf)))
						{
							out.write(buf, 0, n);
						}

						out.close();
						in.close();

						wsdlBindingWsMetadataResourceBlobString = out.toString();

					}
					catch (IOException e)
					{
						info("==> Unable to open specified URL");

						LOGGER.log(Level.SEVERE, e.getMessage(), e);
						System.exit(1);
					}
				}

				if ("FILE_LOCAL".equalsIgnoreCase(wsdlLocatorType))
				{
					info("Attempting to retrieve WSDL/WADL from file : " + wsdlLocator);
					wsdlBindingWsMetadataResourceBlobString = convertXMLFileToString(wsdlLocator);
				}
				else if ("FILE_REMOTE".equalsIgnoreCase(wsdlLocatorType))
				{

					info("Attempting to retrieve WSDL/WADL from URL : " + wsdlLocator);

					try
					{
						URL url = new URL(wsdlLocator);

						InputStream in = new BufferedInputStream(url.openStream());
						ByteArrayOutputStream out = new ByteArrayOutputStream();

						byte[] buf = new byte[1024];
						int n = 0;

						while (-1 != (n = in.read(buf)))
						{
							out.write(buf, 0, n);
						}

						out.close();
						in.close();

						wsdlBindingWsMetadataResourceBlobString = out.toString();

					}
					catch (IOException e)
					{
						info("==> Unable to open specified remote file");

						LOGGER.log(Level.SEVERE, e.getMessage(), e);
						System.exit(1);
					}
				}
				else if ("CLASSPATH_RESOURCE".equalsIgnoreCase(wsdlLocatorType))
				{

					info("Attempting to retrieve WSDL/WADL from file : " + wsdlLocator);

					try
					{
						URL url = ClassLoader.getSystemResource(wsdlLocator);

						InputStream in = new BufferedInputStream(url.openStream());
						ByteArrayOutputStream out = new ByteArrayOutputStream();

						byte[] buf = new byte[1024];
						int n = 0;

						while (-1 != (n = in.read(buf)))
						{
							out.write(buf, 0, n);
						}

						out.close();
						in.close();

						wsdlBindingWsMetadataResourceBlobString = out.toString();

					}
					catch (IOException e)
					{
						info("==> Unable to open specified CLASSPATH resource");

						LOGGER.log(Level.SEVERE, e.getMessage(), e);
						System.exit(1);
					}
				}
			}
		}

		if (wsdlBindingWsMetadataResourceBlobString != null)
		{
			info("");
			info("Retrieved WSDL/WADL :");
			info("");

			if (fullLogging)
			{
				info(wsdlBindingWsMetadataResourceBlobString);
			}

		}
		else
		{

			info("==> Unable to retrieve WSDL/WADL !!");
			System.exit(1);
		}

		wsdlBindingWsMetadataResourceBlob.setBytes(wsdlBindingWsMetadataResourceBlobString.getBytes());
		wsdlBindingWsMetadataResourceBlob.setCode(new Integer(81936)); //NOSONAR: It is compiled by 1.4
		wsdlBindingWsMetadataResourceBlob.setVersion(new Integer(1)); //NOSONAR: It is compiled by 1.4

		// By this point, I have the blob of the retrieved WSDL/WADL

		if (isWadl)
		{
			info("");
			info("- Description type = WADL");
			info("");
		}
		else
		{
			info("");
			info("- Description type = WSDL");
			info("");
		}

		WsMetadata wsdlBindingParseOptionsMetadata = new WsMetadata();
		WsResource wsdlBindingParseOptionsMetadataResource = new WsResource();
		WsResource[] wsdlBindingParseOptionsMetadataResources = new WsResource[1];

		ServiceDescriptionLanguageParseInput sdlInput = new ServiceDescriptionLanguageParseInput();
		sdlInput.setMetadata(wsdlBindingParseOptionsMetadata);
		sdlInput.setReturnXmlTypes(false);
		sdlInput.setReturnMetadata(true);

		wsdlBindingParseOptionsMetadata.setIgnoreProxySettings(Boolean.TRUE);
		wsdlBindingParseOptionsMetadata.setIsMexLocation(Boolean.FALSE);
		wsdlBindingParseOptionsMetadataResource.setCode(new Integer(81936)); //NOSONAR: It is compiled by 1.4
		wsdlBindingParseOptionsMetadataResource.setVersion(new Integer(1)); //NOSONAR: It is compiled by 1.4

		wsdlBindingParseOptionsMetadataResource.setLocation(wsdlLocator);
		wsdlBindingParseOptionsMetadata.setLocation(wsdlLocator);
		wsdlBindingParseOptionsMetadataResource.setContent(wsdlBindingWsMetadataResourceBlob);

		wsdlBindingParseOptionsMetadataResources[0] = wsdlBindingParseOptionsMetadataResource;
		wsdlBindingParseOptionsMetadata.setResources(wsdlBindingParseOptionsMetadataResources);

		if (isWadl)
		{
			wsdlBindingParseOptionsMetadata.setAddFallbackOperations(Boolean.FALSE);
			wsdlBindingParseOptionsMetadata.setExposeAsRest(Boolean.FALSE);

			wsdlBindingParseOptionsMetadataResource.setDialect("wadl");
			wsdlBindingParseOptionsMetadataResource.setNamespace(wsdlBindingQNameXmlNs);
			wsdlBindingParseOptionsMetadata.setBaseWSDL(wsdlBindingParseOptionsMetadataResource);
		}
		else
		{
			wsdlBindingParseOptionsMetadataResource.setDialect("http://schemas.xmlsoap.org/wsdl/");
		}

		WsdlContentDef wsdlBindingContentDef = getConfig().sdlTask.parse(sdlInput);

		boolean validWSDL = false;

		if (wsdlBindingContentDef != null)
		{
			if (wsdlBindingContentDef.getMetadata() != null)
			{
				info("");
				info("- Parsed specified WSDL/WADL!");
				info("");

				validWSDL = true;
			}
		}

		// ////////

		if (!validWSDL)
		{
			info("==> Unable to parse WSDL/WADL !!");
			System.exit(1);
		}

		WsResource wsdlBindingWsMetadataResource = new WsResource();
		WsResource[] wsdlBindingWsMetadataResources = new WsResource[1];
		wsdlBindingWsMetadataResource.setCode(new Integer(81936)); //NOSONAR: It is compiled by 1.4
		wsdlBindingWsMetadataResource.setVersion(new Integer(1)); //NOSONAR: It is compiled by 1.4
		wsdlBindingWsMetadataResource.setContent(wsdlBindingWsMetadataResourceBlob);
		wsdlBindingWsMetadataResource.setLocation(wsdlLocator);
		wsdlBindingWsMetadataResources[0] = wsdlBindingWsMetadataResource;

		if (isWadl)
			wsdlBindingWsMetadataResource.setDialect("wadl");
		else
			wsdlBindingWsMetadataResource.setDialect("http://schemas.xmlsoap.org/wsdl/");

		// ////////

		WsMetadata wsdlBindingWsMetadata = new WsMetadata();

		wsdlBindingWsMetadata.setBaseWSDL(wsdlBindingWsMetadataResource);
		wsdlBindingWsMetadata.setCode(new Integer(81936)); //NOSONAR: It is compiled by 1.4
		wsdlBindingWsMetadata.setResources(wsdlBindingWsMetadataResources);
		wsdlBindingWsMetadata.setVersion(new Integer(1)); //NOSONAR: It is compiled by 1.4

		wsdlBindingWsMetadata.setIgnoreProxySettings(Boolean.TRUE);
		wsdlBindingWsMetadata.setIsMexLocation(Boolean.FALSE);
		wsdlBindingWsMetadata.setLocation(wsdlLocator);

		if ((wsdlBindingContentDef != null) && (wsdlBindingContentDef.getMetadata() != null))
		{
			if (wsdlBindingContentDef.getMetadata().getBaseWSDL() != null)
			{
				wsdlBindingWsMetadata.setBaseWSDL(wsdlBindingContentDef.getMetadata().getBaseWSDL());
				info("- Set BaseWSDL to parsed value");
			}

			if (wsdlBindingContentDef.getMetadata().getResources() != null)
			{
				wsdlBindingWsMetadata.setResources(wsdlBindingContentDef.getMetadata().getResources());
				info("- Set Resources to parsed value(s)");
			}
		}

		// Set temporary values (else WsdlBindingDef retrieval seems to fail!)

		QName wsdlBindingQName = new QName();

		// NOTE :- These values must be valid values from the specified WSDL/WADL else this will error.

		info("");
		info("==> Specified QName (LocalName) : " + wsdlBindingQNameLocalName);
		info("==> Specified QName (XmlNs) : " + wsdlBindingQNameXmlNs);
		info("");

		wsdlBindingQName.setLocalName(wsdlBindingQNameLocalName);
		wsdlBindingQName.setXmlNs(wsdlBindingQNameXmlNs);

		// ////////

		WsdlBinding wsdlBinding = new WsdlBinding();

		wsdlBinding.setCode(new Integer(81936)); //NOSONAR: It is compiled by 1.4
		wsdlBinding.setMetadata(wsdlBindingWsMetadata);
		wsdlBinding.setQName(wsdlBindingQName);
		wsdlBinding.setVersion(new Integer(1)); //NOSONAR: It is compiled by 1.4

		WsdlBindingDef wsdlBindingDef = null;

		info("Parsing WsdlBinding...");

		wsdlBindingDef = getConfig().sdlTask.getSdlBinding(wsdlBinding);

		// displayWsdlBindingDef(wsdlBindingDef);

		if (wsdlBindingDef != null)
		{
			info("Parsing WsdlBindingDef...");

			// Determine certain values from the supplied WSDL/WADL

			QName parsedWsdlBindingQName = new QName();

			parsedWsdlBindingQName = getQNameFromWsdlBindingDef(wsdlBindingDef);
			// displayQName(parsedWsdlBindingQName);

			if (parsedWsdlBindingQName != null)
			{
				wsdlBindingQName = new QName();

				wsdlBindingQName.setLocalName(parsedWsdlBindingQName.getLocalName());
				wsdlBindingQName.setXmlNs(parsedWsdlBindingQName.getXmlNs());

				wsdlBinding.setQName(wsdlBindingQName);

				return wsdlBinding;
			}
			else
			{
				info("Cannot retrieve QName details : QName appears to be null!");
			}
		}
		else
		{
			info("Cannot determine QName : wsdlBindingDef appears to be null!");
		}

		return null;
	}

	protected String convertXMLFileToString(String filePath) throws Exception
	{

		if (filePath.length() > 0)
		{
			try
			{
				DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();

				try (InputStream inputStream = new FileInputStream(new File(filePath))) {
					Document doc = documentBuilderFactory.newDocumentBuilder().parse(inputStream);

					StringWriter stw = new StringWriter();
					Transformer serializer = TransformerFactory.newInstance().newTransformer();

					serializer.transform(new DOMSource(doc), new StreamResult(stw));

					return stw.toString();
				}
			}
			catch (Exception e)
			{
				LOGGER.log(Level.SEVERE, e.getMessage(), e);
			}
		}

		return null;
	}

	protected QName getQNameFromWsdlBindingDef(WsdlBindingDef wsdlDef) throws Exception
	{
		if (wsdlDef != null)
		{
			if (wsdlDef.getQName() != null)
			{
				return wsdlDef.getQName();
			}
		}
		else
		{
			info("Invalid WsdlBindingDef specified");
		}

		return null;
	}

	/** <!-- ================================================================================================== -->
	 * Convert the QName to a string
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected String toString(QName qname) throws Exception
	{
		if (qname == null)
			return null;

		return Util.toString(qname.getLocalName()) + "#" + Util.toString(qname.getXmlNs());
	}

	/** This is the output of the WSDL/WADL choice. */
	protected static class WsdlChoice
	{
		WsdlBinding binding;

		WsdlBindingDef bindingDef;

		WsdlPortDef portDef;
	}

	/** <!-- ================================================================================================== -->
	 * Prompt the User for a WSDL/WADL and select the port.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected WsdlChoice promptForSdl(String defaultSDLURL) throws Exception
	{
		// ---------------------------------------------------------
		// Find the WSDL/WADL and prompt for port selection if
		// necessary.
		// ---------------------------------------------------------

		// One of the following must be entered.
		String sdlPath = Input.read("Local WSDL/WADL path [For URL hit enter for next prompt]");
		String sdlURL = null;
		WsdlContentDef parsedSdl = null;

		// This uses SOAPstation to parse the contents of the WSDL/WADL file.
		info("Parsing WSDL/WADL...");

		if (Util.isBlank(sdlPath))
		{
			sdlURL = Input.read("WSDL/WADL URL", defaultSDLURL);
			parsedSdl = getSdlAPITask().parseSDL(sdlURL, false);
		}
		else
		{
			byte[] content = Util.readFileAsBytes(sdlPath);
			parsedSdl = getSdlAPITask().parseSDL(content);
		}

		// Check how many ports and bindings there are in the WSDL/WADL.
		// If there are more than one, then we have to prompt the user
		// to know which one they want to manage.
		WsdlTask.WsdlBindingPortDef selectedBindingPort = null;
		List bindingPorts = getSdlAPITask().expandPortsFromBindings(parsedSdl.getBindings());
		int numBindingPorts = Util.size(bindingPorts);

		if (numBindingPorts <= 0)
		{
			// The wsdl did not have any binding + port pairs.
			// TODO: Prompt the user to select a binding and specify the port
			// where the service is implemented.
			throw new Exception("The WSDL/WADL did not contain any ports for the bindings.");
		}
		else if (numBindingPorts == 1)
		{
			// There is only one port. Select it automatically.
			selectedBindingPort = (WsdlTask.WsdlBindingPortDef) bindingPorts.get(0);
		}
		else
		{
			// Prompt the user on which port they want to use
			// for the service group.
			info("WSDL/WADL contains multiple ports and / or multiple bindings:");
			for (int i = 0; i < numBindingPorts; i++)
			{
				WsdlTask.WsdlBindingPortDef bindingPort = (WsdlTask.WsdlBindingPortDef) bindingPorts
						.get(i);

				info(	1, 
					i + ") " + toString(bindingPort.binding.getQName()) + " - " + 
							bindingPort.port.getName());
			}

			int choice = Input.readInteger("Choose one", new Integer(0)); //NOSONAR: It is compiled by 1.4
			selectedBindingPort = (WsdlTask.WsdlBindingPortDef) bindingPorts.get(choice);
		}

		// Create the WSDLBinding
		WsdlBinding wsdlBinding = new WsdlBinding();

		wsdlBinding.setQName(selectedBindingPort.binding.getQName());
		wsdlBinding.setMetadata(parsedSdl.getMetadata());

		// Build the response
		WsdlChoice choice = new WsdlChoice();

		choice.binding = wsdlBinding;
		choice.bindingDef = selectedBindingPort.binding;
		choice.portDef = selectedBindingPort.port;

		return choice;
	}

	/** <!-- ================================================================================================== -->
	 * Create WebService
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected String createWebService() throws Exception
	{
		info("---------");
		info("Creating Service Group...");
		String svcGroupName = Input.read("Name", "SvcGroup");
		WsdlChoice wsdl = promptForSdl("http://ws.cdyne.com/ip2geo/ip2geo.asmx?wsdl");
		Endpoint endpoint = getSdlAPITask().buildEndpoint(wsdl.portDef);

		// ---------------------------------------------------------
		// Prompt for the application to use.
		// ---------------------------------------------------------

		// Get a list of applications configured by user
		info("Configured Applications:");
		Application application = selectApplication("Select application for new Group.");
		application = (Application) Util.createLeaf(application);

		// ---------------------------------------------------------
		// Select a listener / sender
		// ---------------------------------------------------------

		// This code is assuming that the same Transport can be
		// used both as a listener and a sender.

		// We are assuming that the same transport will be used as a listener and sender
		// for the newly created service group.
		info("Existing transports:");
		Transport transport = (Transport) Util
				.createLeaf(selectExistingTransport("Select Transport to use as the listener / "
						+ "sender for the new service group"));
		transport = (Transport) Util.createLeaf(transport);

		// ---------------------------------------------------------
		// Create the service group.
		// ---------------------------------------------------------

		// Set the service group name
		SvcGroup svcGroup = new SvcGroup();
		svcGroup.setKeyName(svcGroupName);

		// Set the Listener
		// AccessPointNode apNode = new AccessPointNode();
		// apNode.setTransport(transport);

		// Set the Sender
		// ManagedServiceNode msNode = new ManagedServiceNode();

		// Create access control rule to grant access??
		// Change default Access Control Behavior to deny

		String svcGroupID = getConfig().serviceTask.addSimpleServiceGroup(wsdl.binding, wsdl.portDef,
				new Object[] { svcGroup,
						// apNode,
						// msNode,
						application, transport, null, null, endpoint });

		// Activate the SvcGroup right away.
		getConfig().serviceTask.activateServiceGroupRevision(svcGroupID);
		return svcGroupID;
	}

	protected String createRESTService() throws Exception
	{
		info("---------");
		info("Creating REST Service Group...");
		String svcGroupName = Input.read("Name", "REST SvcGroup");
		WsdlChoice wsdl = promptForSdl("http://www.herongyang.com/Service/Hello_REST.wadl");
		Endpoint endpoint = getSdlAPITask().buildEndpoint(wsdl.portDef);

		String[] ops = selectSvcOperations(wsdl.binding);

		// ---------------------------------------------------------
		// Prompt for the application to use.
		// ---------------------------------------------------------

		// Get a list of applications configured by user
		info("Configured Applications:");
		Application application = selectApplication("Select application for new Group.");
		application = (Application) Util.createLeaf(application);

		// ---------------------------------------------------------
		// Select a listener / sender
		// ---------------------------------------------------------

		// This code is assuming that the same Transport can be
		// used both as a listener and a sender.

		// We are assuming that the same transport will be used as a listener and sender
		// for the newly created service group.
		info("Existing transports:");
		Transport transport = (Transport) Util
				.createLeaf(selectExistingTransport("Select Transport to use as the listener / "
						+ "sender for the new service group"));
		transport = (Transport) Util.createLeaf(transport);

		// ---------------------------------------------------------
		// Create the service group.
		// ---------------------------------------------------------

		// Set the service group name
		SvcGroup svcGroup = new SvcGroup();
		svcGroup.setKeyName(svcGroupName);

		// Set the Listener
		// AccessPointNode apNode = new AccessPointNode();
		// apNode.setTransport(transport);

		// Set the Sender
		// ManagedServiceNode msNode = new ManagedServiceNode();

		// Create access control rule to grant access??
		// Change default Access Control Behavior to deny

		String svcGroupID = getConfig().serviceTask.addSimpleServiceGroup(wsdl.binding, wsdl.portDef,
				new Object[] { svcGroup,
						// apNode,
						// msNode,
						application, transport, ops, null, endpoint });

		// Activate the SvcGroup right away.
		getConfig().serviceTask.activateServiceGroupRevision(svcGroupID);
		return svcGroupID;
	}

	/** <!-- ================================================================================================== -->
	 * Modify the specified service group.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public void updateWebservice(String svcGroupID) throws Exception
	{
		info("---------");
		info("Creating a new revision of group: " + svcGroupID);
		// create a new revision and modify it.
		svcGroupID = getConfig().serviceTask.addServiceGroupRevision(svcGroupID);

		SvcGroup svcGroup = (SvcGroup) getConfig().configTask.get(svcGroupID);

		info("Modifying service group: " + svcGroup.getKeyName());

		// ---------------------------------------------------------
		// Prepare the structures to modify
		// ---------------------------------------------------------

		// Pick up the first MS, and AP.
		AccessPointNode apNode = getConfig().serviceTask.getAccessPoints(svcGroupID)[0];
		ManagedServiceNode msNode = getConfig().serviceTask.getManagedServices(svcGroupID)[0];

		msNode = (ManagedServiceNode) getConfig().acervate(msNode);

		// Split the AP into the different parts so that we can edit it easier.
		// This will accervate the Node.
		AccessPointDef apDef = getConfig().apTask.decomposeNode(apNode);
		apNode = apDef.getNode();

		// ---------------------------------------------------------
		// Prompt for the new SvcGroup Name
		// ---------------------------------------------------------

		String groupName = Input.read("New Name", "SvcGroup_Renamed");
		svcGroup.setKeyName(groupName);

		// ---------------------------------------------------------
		// Change the WSDL/WADL
		// ---------------------------------------------------------

		// Get the new WSDL and update it in 3 places.
		WsdlChoice newWSDL = promptForSdl("http://api.google.com/GoogleSearch.wsdl");

		apNode.setWsdlBinding(newWSDL.binding);
		msNode.setWsdlBinding(newWSDL.binding);
		svcGroup.setIntermediateWsdl(newWSDL.binding);

		// Clear the list of operations in the AP
		// so that SST picks up the new list from
		// the WSDL. The MS and intermediate nodes
		// always update their list during an edit.
		apNode.setSvcOps(null);

		// If the user wants to filter the operations
		// exposed, then remove the hidden operations
		// from this array.
		// SvcOperation[] allOps = getConfig().wsdlTask.createSvcOperationsForBindingDef(newWSDL.bindingDef);
		// apNode.setSvcOps(allOps);

		// change the endpoint
		msNode.setEndpoint(getSdlAPITask().buildEndpoint(newWSDL.portDef));

		// Get the suggested xml processor
		XmlProcessorLC xmlProcessor = getConfig().wsdlTask.suggestXmlProcessor(newWSDL.bindingDef);

		// Compare - Check the KeyID
		XmlProcessorLC existingXmlProcessor = apNode.getXmlProcessor();

		// Get the chosen XML processor
		XmlProcessorLC chosenProc = choseXmlProcessor(xmlProcessor, existingXmlProcessor);

		// Update the xml processor
		apNode.setXmlProcessor(chosenProc);
		msNode.setXmlProcessor(chosenProc);
		svcGroup.setXmlProcessor(chosenProc);

		// ---------------------------------------------------------
		// Select a listener / sender
		// ---------------------------------------------------------

		// This code is assuming that the same Transport can be
		// used both as a listener and a sender.

		// We are assuming that the same transport will be used as a listener and sender
		// for the newly created service group.
		info("Existing transports:");
		Transport transport = (Transport) Util
				.createLeaf(selectExistingTransport("Select Transport to use as the listener / "
						+ "sender for the modified service group"));
		transport = (Transport) Util.createLeaf(transport);
		apNode.setTransport(transport);

		msNode.getEndpoint().setTransport(transport);

		// ---------------------------------------------------------
		// Prompt for the application to use.
		// ---------------------------------------------------------

		// Get a list of applications configured by user
		info("Configured Applications:");
		Application application = selectApplication("Select application for new Group.");
		application = (Application) Util.createLeaf(application);
		msNode.setApplication(application);

		// ---------------------------------------------------------
		// Commit the changes.
		// ---------------------------------------------------------

		getConfig().serviceTask.updateAccessPoint(apNode);
		getConfig().serviceTask.updateManagedService(msNode);
		getConfig().serviceTask.updateServiceGroup(svcGroup);

		getConfig().serviceTask.activateServiceGroupRevision(svcGroupID);
	}

	/** <!-- ================================================================================================== -->
	 * Find the chosen xml processor and return it for update.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */
	public XmlProcessorLC choseXmlProcessor(XmlProcessorLC newProc, XmlProcessorLC prevProc) throws Exception
	{
		info("---------");
		info("Checking for XML Processors....");

		// Comparing and logging message if the xml processors are different
		if (!Util.equals(Util.getKeyID(newProc), Util.getKeyID(prevProc)))
		{
			String existingXmlProcessorName = Util.getKeyName(prevProc);
			String xmlProcessorName = Util.getKeyName(newProc);

			info("XML processor at access point : " + existingXmlProcessorName);
			info("New XML processor : " + xmlProcessorName);
		}
		else
		{
			info("Same XML processor at access point and new WSDL : " + Util.getKeyName(prevProc));
		}

		// Find and display all existing XMLProcessors
		JavaPluginLC[] xmlProcessors = getConfig().javaPluginTask.getPlugins(JavaPluginType.XML_PROCESSOR);

		info("---------");
		info("Listing XML Processors....");

		int suggestedInd = 0;
		for (int i = 0, size = Util.length(xmlProcessors); i < size; i++)
		{
			XmlProcessorLC proc = (XmlProcessorLC) xmlProcessors[i];
			String message = i + ") " + Util.getKeyName(proc);

			// Compare and find number of sugested one
			if (Util.equals(Util.getKeyID(newProc), Util.getKeyID(proc)))
			{
				suggestedInd = i;
				info(i + ") " + Util.getKeyName(proc) + " (suggested)");
			}
			else info(message);
		}

		// Make the default be the sugested one.
		int choice = Input.readInteger("Choose the XML processor", new Integer(suggestedInd)); //NOSONAR: It is compiled by 1.4
		info("---------");

		return (XmlProcessorLC) xmlProcessors[choice];
	}

	/** <!-- ================================================================================================== -->
	 * Find a service group based on name and display the details.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public void findServiceGroup() throws Exception
	{
		info("---------");
		String groupName = Input.read("Enter the name of the Service Group to find", "SvcGroup");

		// ---------------------------------------------------------
		// Create a template of the structure we are trying to find.
		// ---------------------------------------------------------
		SvcGroup tGroup = new SvcGroup();

		// Match the entire service group name example
		tGroup.setKeyName(groupName);

		// To match all service groups that start with the specified string the match needs to
		// be changed to a Regular Expression match (Add ~ to start of string) and then the regex
		// built up.
		// tGroup.setKeyName("~" + groupName + ".*");

		// This field is set to limit the match to active revisions.
		tGroup.setActiveRevision(new SvcGroupRevisions());

		// ---------------------------------------------------------
		// Perform a match for all service groups that match the specified pattern.
		// ---------------------------------------------------------
		BaseData[] matches = getConfig().configTask.find(tGroup);
		int numMatches = Util.length(matches);

		if (numMatches == 0)
		{
			info("No matches found.");
			return;
		}

		// ---------------------------------------------------------
		// Display the details of the groups we found.
		// ---------------------------------------------------------

		for (int i = 0; i < numMatches; i++)
		{
			info("Match " + i + ":");
			displaySvcGroup((SvcGroup) matches[i]);
		}
	}

	public void listKeyCertificates() throws Exception
	{
		CertificateEntry[] certs = getConfig().certTask.getAllKeyCertificateEntries();
		getConfig().acervate(certs);
		Arrays.sort(certs, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(certs); i < size; i++)
		{
			CertificateEntry cert = certs[i];

			info(i + ") " + cert.getKeyName());
		}
	}

	public CertificateEntry selectsKeyCertificate(String prompt) throws Exception
	{
		CertificateEntry[] certs = getConfig().certTask.getAllKeyCertificateEntries();
		getConfig().acervate(certs);
		Arrays.sort(certs, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		listKeyCertificates();
		info(Util.length(certs) + ") Quit");

		int choice = Input.readInteger(prompt, new Integer(0)); //NOSONAR: It is compiled by 1.4

		if (choice < Util.length(certs))
			return (certs[choice]);
		else return null;
	}

	public void displayKeyCertificates() throws Exception
	{
		info("---------");
		info("Certificate details.");

		boolean quit = false;

		do
		{
			CertificateEntry cert = selectsKeyCertificate("Details for Certificate");

			if (cert != null)
				displayCertificateInfo(cert.getKeyID());
			else quit = true;
		}
		while (!quit);
	}

	private void displayCertificateInfo(Blob cert, Credentials cred, CertificateEncodingType encoding)
			throws Exception
	{
		displayCertificateInfo(getConfig().certTask.getKeyCertificateInfo(cert, cred, encoding));
	}

	private void displayCertificateInfo(String entry) throws Exception
	{
		displayCertificateInfo(getConfig().certTask.getCertificateInfoFromID(entry));
	}

	private void displayCertificateInfo(CertificateDef opts) throws Exception
	{
		System.out.println("============= Certificate Information ==================");
		infoField(2, "Version", opts.getVersion());
		infoField(2, "Serial Number", opts.getSerialNumber());
		infoField(2, "Signature Algorithm", opts.getSignatureAlgorithm());
		infoField(2, "Common Name", opts.getCommonName());
		infoField(2, "Organization", opts.getOrganization());
		infoField(2, "Organization Unit", opts.getOrganizationalUnit());
		infoField(2, "Locality", opts.getLocality());
		infoField(2, "State", opts.getState());
		infoField(2, "Country", opts.getCountry());
		infoField(2, "Email", opts.getEmailAddress());
		infoField(2, "Issuer", opts.getIssuer());
		infoField(2, "Valid From", opts.getValidFrom());
		infoField(2, "Valid To", opts.getValidTo());
	}

	public String createRole() throws Exception
	{
		AuthenticatorLC prov = selectAuthenticationPlugin("Select the User Directory");
		UserEntity selectedEntity = selectUserEntity(prov, "Select the user");
		UserEntity[] entities = new UserEntity[1];
		entities[0] = selectedEntity;

		SSTRole role = new SSTRole();
		role.setKeyName("Test user role: " + prov.getKeyName() + "-" + selectedEntity.getKeyName());
		role.setDescription("SDK API created role");
		role.setUserDirectory(prov);
		role.setAllowedEntities(entities);
		return getConfig().roleTask.createRole(role);
	}

	public void createRoles() throws Exception
	{
		AuthenticatorLC prov = selectAuthenticationPlugin("Select the User Directory");
		UserEntity selectedEntity = selectUserEntity(prov, "Select the user");
		UserEntity[] entities = new UserEntity[1];
		entities[0] = selectedEntity;

		SSTRole role = new SSTRole();
		role.setKeyName("Test user role");
		role.setDescription("SDK API created role");
		role.setUserDirectory(prov);
		role.setAllowedEntities(entities);
		getConfig().roleTask.createRole(role);

		SSTRole role2 = new SSTRole();
		role2.setKeyName("Test user filtering role");
		role2.setDescription("SDK API created user filtering role");
		role2.setUserDirectory(prov);
		role2.setUserMembershipFilter("sn=thien");
		getConfig().roleTask.createRole(role2);
	}

	public UserEntity selectUserEntity(AuthenticatorLC prov, String prompt) throws Exception
	{
		UserEntity[] users = getConfig().dirTask.getUserEntities(prov.getKeyID(), "*", DirectoryFilterType.UID);
		getConfig().acervate(users);
		Arrays.sort(users, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(users); i < size; i++)
		{
			UserEntity user = users[i];
			String message = i + ") " + user.getKeyName();

			info(message);
		}

		int choice = Input.readInteger(prompt, new Integer(0)); //NOSONAR: It is compiled by 1.4
		return users[choice];

	}

	/** <!-- ================================================================================================== -->
	 * Select an authentication plugin
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public AuthenticatorLC selectAuthenticationPlugin(String prompt) throws Exception
	{
		AuthenticatorLC[] provs = getConfig().dirTask.getAllUserDirectory();
		getConfig().acervate(provs);
		Arrays.sort(provs, com.actional.soapapi.Util.COMPARATOR_BASEDATA_KEYNAME);

		for (int i = 0, size = Util.length(provs); i < size; i++)
		{
			AuthenticatorLC prov = provs[i];
			String message = i + ") " + prov.getKeyName();

			info(message);
		}

		int choice = Input.readInteger(prompt, new Integer(0)); //NOSONAR: It is compiled by 1.4
		return provs[choice];
	}

	/** <!-- ================================================================================================== -->
	 * Start of program.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public static void main(String[] args)
	{
		// Prompt for User Name, password and sst URL.
		try
		{
			System.out.println("");

			String endpoint = Input.read("SOAPstation address", "http://localhost:4400/sst");
			String username = Input.read("UserName", "admin");
			String password = Input.read("Password", true, "secret");

			System.out.println("");
			System.out.println("Initializing connection...");

			SSTConfig sstConfig = new SSTConfig(username, password, endpoint);
			SSTClient client = new SSTClient(sstConfig);

			client.init();
			client.test();
		}
		catch (Exception e)
		{
			LOGGER.log(Level.SEVERE, e.getMessage(), e);
			System.exit(-1);
		}
	}
}
