//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi.javaClient;

import java.util.logging.Level;
import java.util.logging.Logger;

// From the Task API
import com.actional.lg.db.LGNode;
import com.actional.lg.server.broker.ActiveAgentTO;
import com.actional.soapapi.Util;
import com.actional.soapapi.server.AgentConfigTask;
import com.actional.soapapi.server.ServerConfig;
import com.actional.lg.server.broker.ProvisionInput;

/** <!-- ========================================================================================================== -->
* This is an example of using the SDK to perform different operations.
*
* @lastrev fix41703 - squid:S1148 : Throwable.printStackTrace(...) should not be called
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class ServerClient extends ClientBase
{
	private static final Logger LOGGER = Logger.getLogger(ServerClient.class.getName());
	private static String	DEFAULT_AGENT_NAME = "LocalHost";

	protected AgentConfigTask itsAgentConfigTask;

	/** <!-- ================================================================================================== -->
	 * Constructor
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public ServerClient(ServerConfig serverConfig)
	{
		super(serverConfig);
		itsAgentConfigTask = new AgentConfigTask();
		itsAgentConfigTask.setConfig(serverConfig);
		itsAgentConfigTask.setLog(getLog());
	}

	/** <!-- ================================================================================================== -->
	* Retrieve the config object.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected ServerConfig getConfig()
	{
		return (ServerConfig)itsConfig;
	}

	/** <!-- ================================================================================================== -->
	* Retrieve the agent config object.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected AgentConfigTask getAgentConfigTask()
	{
		return itsAgentConfigTask;
	}

	/** <!-- ================================================================================================== -->
	 * Provision the specified agent.
	 *
	 * @lastrev fix40813 - update I*Agent*Task APIs for QA
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void promptAndProvisionAgent() throws Exception
	{
		String agentMemberId = Input.read("Enter the member id of the agent to provision");

		try
		{
			final ProvisionInput input = new ProvisionInput();

			input.setMemberIDs(new String[] { agentMemberId });
			input.setCompletionTimeout(20L * 1000L);

			getConfig().activeAgentTask.provision(input);
			info("SUCCESS");
		}
		catch (Exception e)
		{
			info("FAILED: Agent provisioning failed. Reason: " + e.getMessage());
		}
	}

	/** <!-- ================================================================================================== -->
	 * List all of the nodes in the system. If there are a lot of nodes, this can be a slow operation.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listNodes() throws Exception
	{
		info("Listing nodes...");

		listNodes(getConfig().lgCallSiteTask.getAllLGNodes());
	}

	/** <!-- ================================================================================================== -->
	 * List all of the nodes with agents.
	 *
	 * @lastrev fix40692 - Old task APIs related to manually managing an Agent should be removed
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listAgents() throws Exception
	{
		info("Listing active agents...");

		listAgents(getConfig().activeAgentTask.getAllActiveAgents());
	}

	/** <!-- ================================================================================================== -->
	 * List the specified nodes
	 *
	 * @lastrev fix41203 - skeleton for IProcessTask
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listAgents(ActiveAgentTO[] agents) throws Exception
	{
		int numMatches = Util.length(agents);

		info("Found " + numMatches + " agents");

		for (int i = 0; i < numMatches; i++)
		{
			ActiveAgentTO agent = agents[i];

			StringBuilder sb = new StringBuilder(64);
			sb.append("- ");

			sb.append("Member Id:");
			sb.append(agent.getAppID());
			sb.append(", Profile Name");
			sb.append(agent.getProfileName());
			sb.append(", Hostname: ");
			sb.append(agent.getHostName());

			info(sb.toString());
		}
	}

	/** <!-- ================================================================================================== -->
	 * List the specified nodes
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void listNodes(LGNode[] nodes) throws Exception
	{
		int numMatches = Util.length(nodes);

		info("Found " + numMatches + " nodes (M = managed, U = unmanaged):");

		for (int i = 0; i < numMatches; i++)
		{
			LGNode node = (LGNode)nodes[i];

			StringBuffer sb = new StringBuffer(64);
			sb.append("- ");

			if (getAgentConfigTask().isManaged(node))
				sb.append("[M] ");
			else
				sb.append("[U] ");

			sb.append(node.getKeyName());

			info(sb.toString());
		}
	}

	/** <!-- ================================================================================================== -->
	 * Display a menu and perform and action.
	 *
	 * @lastrev fix41703 - squid:S1148 : Throwable.printStackTrace(...) should not be called
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public void promptForInputAndPerformAction() throws Exception
	{
		while (true)
		{
			try
			{
				info("");
				info("-----------------------------------------");
				info("Select action to perform:");
				info("- [Pp] Provision agent.");
				info("- [Ll] List nodes.");
				info("- [Aa] List agents.");
				info("- [Cc] Check nodes.");
				info("- [Vv] Version");
				info("- [Qq] Quit.");

				String input = Input.read("Choice");

				info("");

				if ((input == null) || (input.length() != 1))
					continue;

				char a = input.charAt(0);

				switch (a)
				{

					case 'P': case 'p':
						promptAndProvisionAgent();
						break;

					case 'L': case 'l':
						listNodes();
						break;

					case 'A': case 'a':
						listAgents();
						break;

					case 'C': case 'c':
						getAgentConfigTask().checkNodes();
						break;

					case 'V': case 'v':
						displayVersion();
						break;

					case 'Q': case 'q':
						return;

					default:
						info("Invalid action: " + input);
						break;
				}
			}
			catch (Exception e)
			{
				info("Error: " + e.toString());
				LOGGER.log(Level.SEVERE, e.getMessage(), e);
			}
		}
	}

	/** <!-- ================================================================================================== -->
	 * Start of program.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public static void main(String[] args)
	{
		// Prompt for User Name, password and Server URL.
		try
		{
			System.out.println("");

			String endpoint = Input.read("Server address", "http://localhost:4040/lgserver");
			String username = Input.read("UserName", "User_Admin");
			String password = Input.read("Password", true, "security");

			System.out.println("");
			System.out.println("Initializing connection...");

			ServerConfig serverConfig = new ServerConfig(username, password, endpoint);
			ServerClient client = new ServerClient(serverConfig);

			client.init();
			client.promptForInputAndPerformAction();
		}
		catch (Exception e)
		{
			LOGGER.log(Level.SEVERE, e.getMessage(), e);
			System.exit(-1);
		}
	}
}